<?php
namespace Reewards\Domain;

if (!defined('ABSPATH')) exit;

class TirageService
{
    /**
     * Logger interne (option + error_log)
     */
    public static function log($message) {
        $time = current_time('d-m-Y H:i:s');
        $log_entry = "[$time] " . sanitize_text_field($message);
        $logs = get_option('rewaardsc_cron_logs', []);
        $logs[] = $log_entry;
        if (count($logs) > 500) {
            $logs = array_slice($logs, -500);
        }
        update_option('rewaardsc_cron_logs', $logs);
    }

    /**
     * Exécute le tirage pour un concours donné
     */
    public static function lancerTirage($post_id) {
        $post_id = intval($post_id);
        $participants = get_post_meta($post_id, 'concours_participants', true);
        if (!is_array($participants)) {
            $participants = maybe_unserialize($participants);
            if (!is_array($participants)) $participants = [];
        }
        $winners_count = intval(get_post_meta($post_id, 'concours_winners_count', true));
        if ($winners_count < 1) $winners_count = 1;

        $nom_concours = esc_html(get_the_title($post_id));
        if (empty($participants)) {
            self::log(
                sprintf(
                    // translators: 1: contest name, 2: contest ID.
                    __('TAS - Aucun participant trouvé pour le concours "%1$s" (ID=%2$d)', 'reewards'),
                    $nom_concours,
                    $post_id
                )
            );
            return;
        }

        $gagnants = self::tirerGagnants($participants, $winners_count);

        // Enregistrer les gagnants
        update_post_meta($post_id, 'concours_gagnants', $gagnants);
        update_post_meta($post_id, 'concours_tirage_effectue', 1);

        // Envoyer le mail aux gagnants
        $rewards = get_post_meta($post_id, 'concours_rewards', true);
        foreach ($gagnants as $gagnant) {
            self::envoyerMailGagnant($post_id, $gagnant, $rewards);
        }

        // Envoyer le mail à l'administrateur
        self::envoyerMailAdmin($post_id, $gagnants);

        // Mail aux perdants
        $gagnant_emails = array_map(function($g) { return isset($g['email']) ? sanitize_email($g['email']) : null; }, $gagnants);
        $gagnant_emails = array_filter($gagnant_emails);

        foreach ($participants as $perdant) {
            $email = isset($perdant['email']) ? sanitize_email($perdant['email']) : null;
            if (!$email || in_array($email, $gagnant_emails)) continue;
            self::envoyerMailPerdant($post_id, $perdant);
        }

        // Ajout dans les logs
        self::log(
            sprintf(
                // translators: 1: contest name, 2: contest ID, 3: number of winners.
                __('Tirage effectué pour le concours nom = %1$s ID = %2$d. Nombre de gagnants : %3$d', 'reewards'),
                $nom_concours,
                $post_id,
                count($gagnants)
            )
        );
    }

    /**
     * Tirage des gagnants sans doublons
     */
    public static function tirerGagnants($participants, $winners_count) {
        shuffle($participants);
        return array_slice($participants, 0, $winners_count);
    }

    /**
     * Envoi email à l'admin avec les gagnants
     */
    public static function envoyerMailAdmin($post_id, $gagnants) {
        $email_templates = get_option('rewaardsc_emails_templates', []);
        $admin_email_settings = isset($email_templates['admin']) ? $email_templates['admin'] : [];

        $is_active = isset($admin_email_settings['active']) ? intval($admin_email_settings['active']) : 1;
        if ($is_active !== 1) {
            return;
        }

        $subject = !empty($admin_email_settings['title'])
            ? sanitize_text_field($admin_email_settings['title'])
            : __('Notification de concours', 'reewards');

        $body_content = !empty($admin_email_settings['content'])
            ? sanitize_textarea_field($admin_email_settings['content'])
            : __('Le tirage au sort a été effectué.', 'reewards');
        

        $admin_email = sanitize_email(get_option('rewaardsc_admin_email'));
        if (!$admin_email) {
            self::log(__("Aucun email admin configuré, impossible d'envoyer le mail.", 'reewards'));
            return;
        }

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $sent = wp_mail($admin_email, $subject, $body_content, $headers);

        if ($sent) {
            self::log(
            sprintf(
                // translators: 1: admin email address, 2: contest ID.
                __('Mail envoyé à %1$s pour le concours ID=%2$d', 'reewards'),
                $admin_email,
                $post_id
            )
        );
        } else {
            self::log(sprintf(
                // translators: %d is the contest ID.
                __("ERREUR lors de l'envoi du mail pour le concours ID=%d", 'reewards'),
                $post_id
            ));
        }
    }

    /**
     * Envoi email aux gagnants
     */
    public static function envoyerMailGagnant($post_id, $gagnant, $rewards) {
        $email_templates = get_option('rewaardsc_emails_templates', []);
        $winner_email_settings = isset($email_templates['winner']) ? $email_templates['winner'] : [];

        $is_active = isset($winner_email_settings['active']) ? intval($winner_email_settings['active']) : 1;
        if ($is_active !== 1) {
            return;
        }

        $subject = !empty($winner_email_settings['title'])
            ? sanitize_text_field($winner_email_settings['title'])
            : __('Félicitations, tu as gagné le concours', 'reewards');

        $body = !empty($winner_email_settings['content'])
            ? sanitize_textarea_field($winner_email_settings['content'])
            : __('Bravo, tu as été tiré au sort comme gagnant du concours.', 'reewards');


        
        $pseudo = isset($gagnant['pseudo']) ? sanitize_text_field($gagnant['pseudo']) : __('Cher participant', 'reewards');
        $email  = isset($gagnant['email']) ? sanitize_email($gagnant['email']) : null;
        if (!$email) return;

        //Remplacement des balises dynamiques
        $subject = str_replace(['{username}', '{email}'], [$pseudo, $email], $subject);
        $body = str_replace(['{username}', '{email}'], [$pseudo, $email], $body);

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $sent = wp_mail($email, $subject, $body, $headers);

        if ($sent) {
            self::log(
            sprintf(
                // translators: 1: winner email address, 2: contest ID.
                __('Mail de félicitations envoyé à %1$s pour le concours ID=%2$d', 'reewards'),
                $email,
                $post_id
            )
        );
        } else {
            self::log(
                sprintf(
                    // translators: 1: winner email address, 2: contest ID.
                    __('ERREUR lors de l’envoi du mail gagnant à %1$s pour le concours ID=%2$d', 'reewards'),
                    $email,
                    $post_id
                )
            );
        }
    }

    /**
     * Envoi d'email aux perdants
     */
    public static function envoyerMailPerdant($post_id, $perdant) {
        if (!class_exists('\Reewards\Utils\License') || !\Reewards\Utils\License::isProUser()) {
            self::log(__("Tentative d'envoi d'email perdant sans licence PRO.", 'reewards'));
            return;
        }

        $email_templates = get_option('rewaardsc_emails_templates', []);
        $loser_email_settings = isset($email_templates['loser']) ? $email_templates['loser'] : [];
        $is_active = isset($loser_email_settings['active']) ? intval($loser_email_settings['active']) : 1;
        if ($is_active !== 1) {
            self::log(__("Envoi de l'email perdant désactivé dans les réglages.", 'reewards'));
            return;
        }

        $subject = !empty($loser_email_settings['title'])
            ? sanitize_text_field($loser_email_settings['title'])
            : __('Merci pour votre participation', 'reewards');
        $body = !empty($loser_email_settings['content'])
            ? sanitize_textarea_field($loser_email_settings['content'])
            : __('Merci d\'avoir participé au concours.', 'reewards');

        $pseudo = isset($perdant['pseudo']) ? sanitize_text_field($perdant['pseudo']) : __('Cher participant', 'reewards');
        $email  = isset($perdant['email']) ? sanitize_email($perdant['email']) : null;
        if (!$email) {
            self::log(__("Aucun email trouvé pour le perdant.", 'reewards'));
            return;
        }

        $subject = str_replace(['{username}', '{email}'], [$pseudo, $email], $subject);
        $body = str_replace(['{username}', '{email}'], [$pseudo, $email], $body);

        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $sent = wp_mail($email, $subject, $body, $headers);

        if ($sent) {
        self::log(
            sprintf(
                // translators: 1: recipient email, 2: contest ID.
                __('Mail de perdant envoyé à %1$s pour le concours ID=%2$d', 'reewards'),
                $email,
                $post_id
            )
        );
    } else {
        self::log(
            sprintf(
                // translators: 1: recipient email, 2: contest ID.
                __('ERREUR lors de l’envoi du mail perdant à %1$s pour le concours ID=%2$d', 'reewards'),
                $email,
                $post_id
            )
        );
    }
    }
}