<?php
namespace Reewards\Frontend;
use Reewards\Utils\Format;

if (!defined('ABSPATH')) exit;

class HistoriqueShortcode
{
    public static function shortcode($atts) {
        ob_start();

        $main_color = esc_attr(get_option('rewaardsc_theme_color', '#0170B9'));
        $text_color = esc_attr(get_option('rewaardsc_text_color', '#ffffff'));
        $show_participants_count = get_option('rewaardsc_show_participants_count', false) ? 1 : 0;
        ?>
        <div id="rewaardsc-historique-container" 
             data-page="1" 
             data-posts-per-page="5">
            <div class="rewaardsc-historique-items"></div>
            <button id="rewaardsc-load-more"><?php esc_html_e('Afficher plus', 'reewards'); ?></button>
        </div>
        <style>
        .rewaardsc-historique-container { border: 1px solid #ccc; border-radius: 8px; overflow: hidden; }
        .rewaardsc-concours-item { border-bottom: 1px solid #eee; }
        .rewaardsc-concours-item:last-child { border-bottom: none; }
        .rewaardsc-concours-header { background-color: <?php echo esc_html($main_color); ?>; color: <?php echo esc_html($text_color); ?>; padding: 10px 15px; cursor: pointer; display: flex; justify-content: space-between; align-items: center; }
        .rewaardsc-concours-header::after { content: '▼'; font-size: 12px; margin-left: 10px; transition: transform 0.3s ease; }
        .rewaardsc-concours-header.active::after { transform: rotate(-180deg); }
        .rewaardsc-concours-content { display: none; padding: 10px 15px; background-color: #f9f9f9; }
        .rewaardsc-concours-header.active + .rewaardsc-concours-content { display: block; }
        .rewaardsc-concours-header:hover { opacity: 0.9; }
        .rewaardsc-rewards-container { display: flex; flex-wrap: wrap; gap: 10px; margin-top: 5px; }
        .rewaardsc-reward-item { display: flex; align-items: center; background: #fff; border: 1px solid #ccc; border-radius: 6px; padding: 5px 10px; box-shadow: 1px 1px 3px rgba(0,0,0,0.1); }
        .rewaardsc-reward-image { width: 40px; height: 40px; object-fit: cover; border-radius: 4px; margin-right: 8px; }
        .rewaardsc-reward-name { font-weight: 600; margin-right: 5px; }
        .rewaardsc-reward-qty { color: #555; }
        #rewaardsc-load-more { margin: 10px; padding: 8px 15px; cursor: pointer; background: <?php echo esc_html($main_color); ?>; color: <?php echo esc_html($text_color); ?>; border: none; border-radius: 6px; }
        #rewaardsc-load-more:disabled { opacity: 0.5; cursor: default; }
        </style>
        <script>
        (function(){
            const container = document.getElementById('rewaardsc-historique-container');
            const itemsContainer = container.querySelector('.rewaardsc-historique-items');
            const loadMoreBtn = document.getElementById('rewaardsc-load-more');
            let page = parseInt(container.dataset.page);
            const postsPerPage = parseInt(container.dataset.postsPerPage);

            itemsContainer.addEventListener('click', function(e){
                const header = e.target.closest('.rewaardsc-concours-header');
                if(header) {
                    header.classList.toggle('active');
                }
            });

            function loadConcours() {
                loadMoreBtn.disabled = true;
                loadMoreBtn.textContent = '<?php esc_attr_e("Chargement...", "reewards"); ?>';

                fetch('<?php echo esc_url(admin_url("admin-ajax.php")); ?>?action=reewards_history_ajax&page=' + page + '&posts_per_page=' + postsPerPage)
                .then(response => response.text())
                .then(html => {
                    if(html.trim() === '') {
                        if(page == 1 ){
                            container.innerHTML = '<p><?php esc_html_e("Aucun historique pour le moment", "reewards"); ?></p>';
                        }
                        loadMoreBtn.style.display = 'none';
                    } else {
                        itemsContainer.insertAdjacentHTML('beforeend', html);
                        page++;
                        loadMoreBtn.disabled = false;
                        loadMoreBtn.textContent = '<?php esc_html_e("Afficher plus", "reewards"); ?>';
                    }
                });
            }

            loadMoreBtn.addEventListener('click', loadConcours);
            loadConcours();
        })();
        </script>
        <?php
        return ob_get_clean();
    }

    public static function censurerPseudo($pseudo) {
        $pseudo = sanitize_text_field($pseudo);
        $length = mb_strlen($pseudo);
        if ($length <= 2) {
            return str_repeat('*', $length);
        }
        $first = mb_substr($pseudo, 0, 1);
        $last  = mb_substr($pseudo, -1);
        $middle = str_repeat('*', $length - 2);
        return $first . $middle . $last;
    }

    public static function ajax() {
        $censor_winners = get_option('rewaardsc_censor_winners', false);
        $show_participants_count = get_option('rewaardsc_show_participants_count', false);
        $page = isset($_GET['page']) ? intval(wp_unslash($_GET['page'])) : 1; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $posts_per_page = isset($_GET['posts_per_page']) ? intval(wp_unslash($_GET['posts_per_page'])) : 5; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $offset = ($page - 1) * $posts_per_page;

        $concours_query = new \WP_Query([
            'post_type'      => 'rewaards_concours',
            'posts_per_page' => $posts_per_page,
            'offset'         => $offset,
            'post_status'    => 'publish',
            'meta_query'     => [
                [
                    'key'     => 'concours_status',
                    'value'   => 'termine',
                    'compare' => '='
                ]
            ],
            'orderby'        => 'meta_value',
            'meta_key'       => 'concours_end',
            'order'          => 'DESC'
        ]);

        if($concours_query->have_posts()) {
            while($concours_query->have_posts()) : $concours_query->the_post();
                $start        = esc_html(get_post_meta(get_the_ID(), 'concours_start', true) ?: '');
                $end          = esc_html(get_post_meta(get_the_ID(), 'concours_end', true) ?: '');
                $winners      = intval(get_post_meta(get_the_ID(), 'concours_winners_count', true));
                $participants = get_post_meta(get_the_ID(), 'concours_participants', true);
                $participants_count = is_array($participants) ? count($participants) : 0;
                $rewards      = maybe_unserialize(get_post_meta(get_the_ID(), 'concours_rewards', true) ?: []);
                $gagnants     = maybe_unserialize(get_post_meta(get_the_ID(), 'concours_gagnants', true) ?: []);
                ?>
                <div class="rewaardsc-concours-item">
                    <div class="rewaardsc-concours-header">
                        <span ><?php the_title(); ?></span>
                        <span class="rewaardsc-concours-dates">
                            <?php echo esc_html(Format::date($start) . ' - ' . Format::date($end)); ?>
                        </span>
                    </div>
                    <div class="rewaardsc-concours-content">
                        <?php
                        $end_datetime = new \DateTime($end);
                        $now = new \DateTime(current_time('mysql'));
                        $diff = $end_datetime->diff($now);

                        $years = $diff->y;
                        $days  = $diff->d;
                        $hours = $diff->h;

                        $parts = [];
                        if ($years > 0) $parts[] = $years . ' ' . _n('année', 'années', $years, 'reewards');
                        if ($days > 0)  $parts[] = $days . ' ' . _n('jour', 'jours', $days, 'reewards');
                        if ($hours > 0) $parts[] = $hours . ' ' . _n('heure', 'heures', $hours, 'reewards');

                        if (!empty($parts)) {
                            // translators: %s is a time duration like "2 days, 3 hours".
                            echo '<p>⏰️ <b>' . sprintf(esc_html__('Ce concours est terminé depuis %s', 'reewards'), esc_html(implode(', ', $parts))) . '</b></p>';
                        }
                        ?>

                        <p>🏆 <strong><?php esc_html_e('Nombre de gagnants :', 'reewards'); ?></strong> <?php echo esc_html($winners); ?></p>

                        <?php if ($show_participants_count) : ?>
                            <p>👥 <strong><?php esc_html_e('Nombre de participants :', 'reewards'); ?></strong> <?php echo esc_html($participants_count); ?></p>
                        <?php endif; ?>
                        <?php if(!empty($rewards)) : ?>
                            <p>🎁 <strong><?php esc_html_e('Récompenses :', 'reewards'); ?></strong></p>
                            <div class="rewaardsc-rewards-container">
                                <?php foreach($rewards as $reward) : 
                                    $image = isset($reward['image']) ? esc_url($reward['image']) : '';
                                    $name  = isset($reward['nom']) ? esc_html($reward['nom']) : '';
                                    $qty   = isset($reward['qty']) ? intval($reward['qty']) : 0;
                                ?>
                                    <div class="rewaardsc-reward-item">
                                        <?php if($image): ?>
                                            <img src="<?php echo esc_url($image); ?>" alt="<?php echo esc_attr($name); ?>" class="rewaardsc-reward-image" />
                                        <?php endif; ?>
                                        <span class="rewaardsc-reward-name"><?php echo esc_html($name); ?></span>
                                        <span class="rewaardsc-reward-qty">x<?php echo esc_html($qty); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>

                        <?php if(!empty($gagnants)) : ?>
                            <p><strong><?php esc_html_e('Gagnants :', 'reewards'); ?></strong></p>
                            <div class="rewaardsc-rewards-container">
                                <?php foreach($gagnants as $winner) : ?>
                                    <div class="rewaardsc-reward-item">
                                        
                                        <span class="rewaardsc-reward-name">🏅 <?php
                                            if ($censor_winners) {
                                                echo esc_html(self::censurerPseudo($winner['pseudo']));
                                            } else {
                                                echo esc_html($winner['pseudo']);
                                            }
                                        ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php
            endwhile;
            wp_reset_postdata();
        }
        wp_die();
    }
}

// Enregistrement du shortcode et des hooks APRÈS la déclaration de la classe
add_shortcode('reewards_history', [HistoriqueShortcode::class, 'shortcode']);
add_action('wp_ajax_reewards_history_ajax', [HistoriqueShortcode::class, 'ajax']);
add_action('wp_ajax_nopriv_reewards_history_ajax', [HistoriqueShortcode::class, 'ajax']);